import adsk.core, adsk.fusion, adsk.cam, traceback
from typing import Tuple, Optional

# このライブラリは
# Fusionが作成するスクリプトと同じディレクトリに置かれる事を想定している
# 使用する場合には、下記コードにより読み込みパスを追加する
"""
import os, sys, importlib
current_dir = os.path.dirname(os.path.abspath(__file__))
script_dir = os.path.dirname(current_dir)
sys.path.append(script_dir)
from library import lib_util
"""
# ---------------------
## Get Fusion object
# ---------------------
# def get_app():
# def get_design():
# def get_ui():
# def disp_message(msg: str):
"""ユーザーにメッセージボックスを表示"""

# ---------------------
## Component & Sketch Functions
# ---------------------
# def get_root_component() -> Optional[adsk.fusion.Component]:
"""ルートコンポーネントを返す"""
# def get_active_component() -> Optional[adsk.fusion.Component]:
"""現在アクティブなコンポーネントを返す"""
# def create_new_component(name: str = 'New_Component') -> Optional[adsk.fusion.Component]:
"""新しいコンポーネントを作成し、アクティブなコンポーネントの配下に追加"""
# def get_parent_sketch(obj: any) -> Optional[adsk.fusion.Sketch]:
"""指定されたオブジェクトの親スケッチを取得"""
# def create_new_sketch(name: str = 'New_Sketch', plane: str = 'xy') -> Optional[adsk.fusion.Sketch]:
"""指定された平面に新しいスケッチを作成"""
# def is_sketch_mode_active() -> Tuple[bool, Optional[adsk.fusion.Sketch]]:
"""現在スケッチモードがアクティブであるかを確認し、スケッチオブジェクトを返す"""
# def get_sketch() -> Optional[adsk.fusion.Sketch]:
"""アクティブなスケッチを返す、アクティブでない場合は新しいスケッチを作成"""
# /********************************************/
# class ExpressionEvaluator:
"""
    安全に四則演算の文字列を評価するクラス
    evaluator = ExpressionEvaluator()
    var = evaluator.evaluate("50/2+30.2")
    print(f"{var}")
"""


# /////////////////////////////////////////////////////////////
# ---------------------
## Get Fusion object
# ---------------------
def get_app():
    return adsk.core.Application.get()


def get_design():
    app = get_app()
    return adsk.fusion.Design.cast(app.activeProduct)


def get_ui():
    app = get_app()
    return app.userInterface


# ---------------------
## メッセージ表示
# ---------------------
def disp_message(msg: str):
    """ユーザーにメッセージボックスを表示"""
    ui = get_ui()
    if ui:
        ui.messageBox(msg)


# ---------------------
## Component & Sketch Functions
# ---------------------
# /////////////////////////////////////////////////////////////
def get_root_component() -> Optional[adsk.fusion.Component]:
    """ルートコンポーネントを返す"""
    design = get_design()
    return design.rootComponent if design else None


# /////////////////////////////////////////////////////////////
def get_active_component() -> Optional[adsk.fusion.Component]:
    """現在アクティブなコンポーネントを返す"""
    design = get_design()
    return design.activeComponent if design else None


# /////////////////////////////////////////////////////////////
def create_new_component(
    name: str = "New_Component",
) -> Optional[adsk.fusion.Component]:
    """新しいコンポーネントを作成し、アクティブなコンポーネントの配下に追加"""
    component = get_active_component()
    if not component:
        component = get_root_component()

    if not component:
        disp_message(
            "コンポーネントが見つかりません。デザインがアクティブか確認してください。"
        )
        return None

    try:
        occurrences = component.occurrences
        new_occurrence = occurrences.addNewComponent(adsk.core.Matrix3D.create())
        new_component = new_occurrence.component
        new_component.name = name
        return new_component
    except Exception:
        disp_message(
            f"新しいコンポーネントの作成に失敗しました:\n{traceback.format_exc()}"
        )
        return None


# /////////////////////////////////////////////////////////////
def get_parent_sketch(obj: any) -> Optional[adsk.fusion.Sketch]:
    """指定されたオブジェクトの親スケッチを取得"""
    return obj.parentSketch if hasattr(obj, "parentSketch") else None


# /////////////////////////////////////////////////////////////
def create_new_sketch(
    name: str = "New_Sketch", plane: str = "xy"
) -> Optional[adsk.fusion.Sketch]:
    """指定された平面に新しいスケッチを作成"""
    component = get_active_component()
    if not component:
        component = get_root_component()

    if not component:
        disp_message("スケッチを作成するコンポーネントが見つかりません。")
        return None

    plane = plane.lower()
    planes_map = {
        "xy": component.xYConstructionPlane,
        "xz": component.xZConstructionPlane,
        "yz": component.yZConstructionPlane,
    }

    base_plane = planes_map.get(plane)
    if not base_plane:
        disp_message(
            f"サポートされていない平面: '{plane}' 'xy', 'xz', 'yz' のいずれかを選択してください。"
        )
        return None

    try:
        sketches = component.sketches
        sketch = sketches.add(base_plane)
        sketch.name = name
        return sketch
    except Exception:
        disp_message(f"新しいスケッチの作成に失敗しました:\n{traceback.format_exc()}")
        return None


# /////////////////////////////////////////////////////////////
def is_sketch_mode_active() -> Tuple[bool, Optional[adsk.fusion.Sketch]]:
    """現在スケッチモードがアクティブであるかを確認し、スケッチオブジェクトを返す"""
    app = get_app()
    active_object = app.activeEditObject
    # is_instance() を使うことで、より安全に型チェックを行う
    if isinstance(active_object, adsk.fusion.Sketch):
        return True, active_object
    else:
        return False, None


# /////////////////////////////////////////////////////////////
def get_sketch() -> Optional[adsk.fusion.Sketch]:
    """アクティブなスケッチを返す、アクティブでない場合は新しいスケッチを作成"""
    is_active, sketch = is_sketch_mode_active()
    if not is_active:
        sketch = create_new_sketch()
    return sketch


# /////////////////////////////////////////////////////////////

import ast
import operator


class ExpressionEvaluator:
    """
    安全に四則演算の文字列を評価するクラス
        evaluator = ExpressionEvaluator()
        var = evaluator.evaluate("50/2+30.2")
        print(f"{var}")
    """

    def __init__(self):
        # サポートする演算子の定義
        self.operators = {
            ast.Add: operator.add,  # 加算
            ast.Sub: operator.sub,  # 減算
            ast.Mult: operator.mul,  # 乗算
            ast.Div: operator.truediv,  # 除算
            ast.USub: operator.neg,  # 単項マイナス
            # 必要に応じて以下を追加
            # ast.Pow: operator.pow,    # べき乗
            # ast.Mod: operator.mod,    # 剰余
        }

    def evaluate(self, expression):
        """
        文字列式を評価して結果を返す
        Args:
            expression (str): 評価する数式の文字列
        Returns:
            float: 計算結果
        Raises:
            SyntaxError: 式の構文が不正な場合
            ValueError: サポートされていない演算子や式が含まれる場合
        """
        try:
            # 式をパースしてASTを取得
            node = ast.parse(expression, mode="eval").body
            # ASTを評価
            return self._evaluate_node(node)
        except SyntaxError as e:
            raise SyntaxError(f"構文エラー: {expression}") from e
        except Exception as e:
            raise ValueError(f"式の評価中にエラーが発生しました: {e}") from e

    def _evaluate_node(self, node):
        """
        ASTノードを再帰的に評価する
        """
        # 数値リテラルの場合
        if isinstance(node, ast.Constant):  # Python 3.8+
            return node.value
        elif isinstance(node, ast.Num):  # 古いバージョン用
            return node.n

        # 単項演算子の場合（例: -5）
        elif isinstance(node, ast.UnaryOp):
            op = self.operators.get(type(node.op))
            if op is None:
                raise ValueError(
                    f"サポートされていない単項演算子: {type(node.op).__name__}"
                )
            # disp_message(f"{type(self._evaluate_node(node.operand))}")
            return op(self._evaluate_node(node.operand))

        # 二項演算子の場合（例: 3 + 4）
        elif isinstance(node, ast.BinOp):
            op = self.operators.get(type(node.op))
            if op is None:
                raise ValueError(
                    f"サポートされていない二項演算子: {type(node.op).__name__}"
                )

            left = self._evaluate_node(node.left)
            right = self._evaluate_node(node.right)

            # ゼロ除算を回避
            if isinstance(node.op, ast.Div) and right == 0:
                raise ZeroDivisionError("ゼロによる除算")

            # disp_message(f"{type(op(left, right))}")
            return op(left, right)

        # 括弧で囲まれた式の場合
        elif isinstance(node, ast.Expr):
            return self._evaluate_node(node.value)

        else:
            raise ValueError(f"サポートされていない式の種類: {type(node).__name__}")
