import adsk.core, adsk.fusion, adsk.cam, traceback
from typing import Tuple, Optional

# ---------------------
## カスタムクラス
# ---------------------
# class kPoint:
"""点基本クラス"""
# class kSketchPoint(kPoint):
"""スケッチ点クラス"""

# ---------------------
## Sketch Geometry Creation Functions
# ---------------------
# def is_equal_sketch_points(sp1: adsk.fusion.SketchPoint, sp2: adsk.fusion.SketchPoint, tolerance: float = 0.0001) -> bool:
"""2つのスケッチ点が指定された許容差内で同じ位置にあるかどうかを判断"""
# def create_sketch_point(x: float = 0, y: float = 0, z: float = 0, sketch: Optional[adsk.fusion.Sketch] = None) -> Optional[adsk.fusion.SketchPoint]:
"""指定された座標にスケッチ点を作成"""
# def create_line_from_sketchpoints(sp1: adsk.fusion.SketchPoint, sp2: adsk.fusion.SketchPoint, construction: bool = False, fixed: bool = False) -> Optional[adsk.fusion.SketchLine]:
"""2つの既存のスケッチ点から直線を作成"""
# def create_line_from_coordinate(x1: float, y1: float, z1: float, x2: float, y2: float, z2: float, sketch: Optional[adsk.fusion.Sketch] = None, construction: bool = False, fixed: bool = False) -> Optional[adsk.fusion.SketchLine]:
"""座標から直接、スケッチ直線を作成"""
# def create_line_from_points(p1: kPoint, p2: kPoint, sketch: Optional[adsk.fusion.Sketch] = None, construction: bool = False, fixed: bool = False) -> Optional[adsk.fusion.SketchLine]:
"""カスタム `kPoint` オブジェクトからスケッチ直線を作成"""
# def draw_center_line(widthX=100,widthY=100,widthZ=100):
"""センターライン作成"""
# def create_circle_from_center(x,y,z,r,sketch=None,construction: bool = False,fixed: bool = False,):
"""中心とR指定で円を描く"""
# def create_circle_from_3_points
"""３点で円を描く"""
# def create_two_point_rectangle
"""対角２点で長方形を描く"""
# def create_center_point_rectangle
"""中心指定で長方形を描く"""

# /////////////////////////////////////////////////////////////
# 利用するlib_utilは、このライブラリと同じディレクトリ
import os, sys, importlib

current_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(current_dir)
import lib_util

# /////////////////////////////////////////////////////////////


# ---------------------
## カスタム、点基本クラス
# ---------------------
# /////////////////////////////////////////////////////////////
class kPoint:
    """3D空間の点を表す基本クラス"""

    def __init__(self, x: float, y: float, z: float = 0.0):
        self.x = x
        self.y = y
        self.z = z


# ---------------------
## カスタム、スケッチ点クラス
# ---------------------
# /////////////////////////////////////////////////////////////
class kSketchPoint(kPoint):
    """Fusion 360 のスケッチ点として使用される座標を、単位とスケッチ平面に応じて変換するクラス"""

    def __init__(
        self,
        x: float = 0.0,
        y: float = 0.0,
        z: float = 0.0,
        sketch: Optional[adsk.fusion.Sketch] = None,
        unit: str = "mm",
    ):
        unit = unit.lower()
        # 単位変換係数を辞書で管理
        scale_factors = {"mm": 0.1, "cm": 1.0, "m": 10.0}
        scale = scale_factors.get(unit, 1.0)  # 存在しない単位は1.0 (cm) を使用

        if not sketch:
            sketch = (
                lib_util.get_sketch()
            )  # スケッチが指定されていなければ、アクティブなスケッチを取得

        # スケッチ平面に応じて座標系を調整 (XZ平面はY軸が下向き)
        # 参照平面が'XZ'の場合は、y座標を反転させる
        # これはFusion 360の座標系の癖に対応
        # 2025/12/28
        # direction_y = -1.0 if sketch and sketch.referencePlane.name == "XZ" else 1.0
        # referencePlaneがnameプロパティを持っており、かつその値が"XZ"であるかを確認
        is_xz_plane = False
        if sketch and hasattr(sketch.referencePlane, 'name'):
            if sketch.referencePlane.name == "XZ":
                is_xz_plane = True

        direction_y = -1.0 if is_xz_plane else 1.0

        # 単位と座標系による方向を調整
        converted_x = x * scale
        converted_y = y * scale * direction_y
        converted_z = z * scale

        super().__init__(converted_x, converted_y, converted_z)
        self.sketch = sketch
        self.unit = unit
        self.sketch_x = converted_x
        self.sketch_y = converted_y
        self.sketch_z = converted_z
        self.scale = scale


# ---------------------
## Sketch Geometry Creation Functions
# ---------------------
# /////////////////////////////////////////////////////////////
def is_equal_sketch_points(
    sp1: adsk.fusion.SketchPoint,
    sp2: adsk.fusion.SketchPoint,
    tolerance: float = 0.0001,
) -> bool:
    """2つのスケッチ点が指定された許容差内で同じ位置にあるかどうかを判断"""
    # ジオメトリオブジェクトの距離を比較
    distance = sp1.geometry.distanceTo(sp2.geometry)
    return distance < tolerance


# /////////////////////////////////////////////////////////////
def create_sketch_point(
    x: float = 0,
    y: float = 0,
    z: float = 0,
    sketch: Optional[adsk.fusion.Sketch] = None,
) -> Optional[adsk.fusion.SketchPoint]:
    """指定された座標にスケッチ点を作成"""
    if not sketch:
        sketch = lib_util.get_sketch()

    if not sketch:
        lib_util.disp_message(
            "スケッチがアクティブでないか、新しいスケッチの作成に失敗しました。"
        )
        return None

    try:
        # kSketchPoint を使用して単位と平面を考慮した座標を取得
        converted_point = kSketchPoint(x, y, z, sketch)
        # adsk.core.Point3D オブジェクトを作成
        point3d = adsk.core.Point3D.create(
            converted_point.x, converted_point.y, converted_point.z
        )
        # スケッチに点を追加
        sketchPoint = sketch.sketchPoints.add(point3d)
        return sketchPoint
    except Exception:
        lib_util.disp_message(
            f"スケッチ点の作成に失敗しました:\n{traceback.format_exc()}"
        )
        return None


# /////////////////////////////////////////////////////////////
def create_line_from_sketchpoints(
    sp1: adsk.fusion.SketchPoint,
    sp2: adsk.fusion.SketchPoint,
    construction: bool = False,
    fixed: bool = False,
) -> Optional[adsk.fusion.SketchLine]:
    """2つの既存のスケッチ点から直線を作成"""
    # 同じスケッチに属しているかチェック
    if lib_util.get_parent_sketch(sp1) != lib_util.get_parent_sketch(sp2):
        lib_util.disp_message("2つの点は異なるスケッチに属しています。")
        return None

    # 2つの点が同じ位置にあるかチェック
    if is_equal_sketch_points(sp1, sp2):
        lib_util.disp_message("2つの点が同じ位置にあります。直線を作成できません。")
        return None

    sketch = lib_util.get_parent_sketch(sp1)
    if not sketch:
        lib_util.disp_message("親スケッチが見つかりません。")
        return None

    try:
        line = sketch.sketchCurves.sketchLines.addByTwoPoints(sp1, sp2)
        line.isConstruction = construction
        line.isFixed = fixed
        return line
    except Exception:
        lib_util.disp_message(
            f"直線（スケッチ点から）の作成に失敗しました:\n{traceback.format_exc()}"
        )
        return None


# /////////////////////////////////////////////////////////////
def create_line_from_coordinate(
    x1: float,
    y1: float,
    z1: float,
    x2: float,
    y2: float,
    z2: float,
    sketch: Optional[adsk.fusion.Sketch] = None,
    construction: bool = False,
    fixed: bool = False,
) -> Optional[adsk.fusion.SketchLine]:
    """座標から直接、スケッチ直線を作成"""
    if not sketch:
        sketch = lib_util.get_sketch()

    if not sketch:
        return None

    # ヘルパー関数を利用して点を先に作成
    p1 = create_sketch_point(x1, y1, z1, sketch)
    p2 = create_sketch_point(x2, y2, z2, sketch)

    if p1 and p2:
        return create_line_from_sketchpoints(p1, p2, construction, fixed)
    else:
        lib_util.disp_message("直線を作成するための点の作成に失敗しました。")
        return None


# /////////////////////////////////////////////////////////////
def create_line_from_points(
    p1: kPoint,
    p2: kPoint,
    sketch: Optional[adsk.fusion.Sketch] = None,
    construction: bool = False,
    fixed: bool = False,
) -> Optional[adsk.fusion.SketchLine]:
    """カスタム `kPoint` オブジェクトからスケッチ直線を作成"""
    if not sketch:
        sketch = lib_util.get_sketch()

    if not sketch:
        return None

    return create_line_from_coordinate(
        p1.x, p1.y, p1.z, p2.x, p2.y, p2.z, sketch, construction, fixed
    )


# /////////////////////////////////////////////////////////////
# def draw_center_line(widthX=100.0, widthY=100.0, widthZ=100.0):
def draw_center_line(
    widthX=100.0, widthY=100.0, widthZ=100.0, construction=True, fixed=True
):
    """センターライン作成"""
    # lib_util.disp_message('lib_2d_draw_center_line run')
    # widthX = 100 if widthX == 0 else widthX
    # widthY = 100 if widthY == 0 else widthY
    # widthZ = 100 if widthZ == 0 else widthZ
    sketch = lib_util.get_sketch()
    if widthX != 0:
        wx = widthX / 2
        xp1 = kPoint(-wx, 0)
        xp2 = kPoint(wx, 0)
        # create_line_from_points(xp1, xp2, sketch, True, True)  # X Axis
        create_line_from_points(xp1, xp2, sketch, construction, fixed)  # X Axis
    if widthY != 0:
        wy = widthY / 2
        yp1 = kPoint(0, -wy)
        yp2 = kPoint(0, wy)
        # create_line_from_points(yp1, yp2, sketch, True, True)  # Y Axis
        create_line_from_points(yp1, yp2, sketch, construction, fixed)  # Y Axis
    if widthZ != 0:
        wz = widthZ / 2
        zp1 = kPoint(0, 0, -wz)
        zp2 = kPoint(0, 0, wz)
        # create_line_from_points(zp1, zp2, sketch, True, True)  # Z Axis
        create_line_from_points(zp1, zp2, sketch, construction, fixed)  # Z Axis


# /////////////////////////////////////////////////////////////
def create_circle_from_center(
    x,
    y,
    z,
    r,
    sketch=None,
    construction: bool = False,
    fixed: bool = False,
):
    """中心とR指定で円を描く"""
    if not sketch:
        sketch = lib_util.get_sketch()
    # kSketchPointで単位とスケッチ方向など調整
    p1 = kSketchPoint(x, y, z, sketch)
    x = p1.sketch_x
    y = p1.sketch_y
    z = p1.sketch_z
    r = r * p1.scale
    try:
        circles = sketch.sketchCurves.sketchCircles
        circle = circles.addByCenterRadius(adsk.core.Point3D.create(x, y, z), r)
        circle.isConstruction = construction
        circle.isFixed = fixed
        # return circle
        return sketch
    except:
        lib_util.disp_message("この設定では円は作成できません。")
        lib_util.disp_message("Failed:\n{}".format(traceback.format_exc()))
        return None


# /////////////////////////////////////////////////////////////
def create_circle_from_3_points(
    x1,
    y1,
    z1,
    x2,
    y2,
    z2,
    x3,
    y3,
    z3,
    sketch=None,
    construction: bool = False,
    fixed: bool = False,
):
    """３点で円を描く"""
    if not sketch:
        sketch = lib_util.get_sketch()
    # kSketchPointで単位とスケッチ方向など調整
    p1 = kSketchPoint(x1, y1, z1, sketch)
    p2 = kSketchPoint(x2, y2, z2, sketch)
    p3 = kSketchPoint(x3, y3, z3, sketch)
    x1 = p1.sketch_x
    y1 = p1.sketch_y
    z1 = p1.sketch_z
    x2 = p2.sketch_x
    y2 = p2.sketch_y
    z2 = p2.sketch_z
    x3 = p3.sketch_x
    y3 = p3.sketch_y
    z3 = p3.sketch_z
    pnt1 = adsk.core.Point3D.create(x1, y1, z1)
    pnt2 = adsk.core.Point3D.create(x2, y2, z2)
    pnt3 = adsk.core.Point3D.create(x3, y3, z3)
    try:
        circles = sketch.sketchCurves.sketchCircles
        circle = circles.addByThreePoints(pnt1, pnt2, pnt3)
        circle.isConstruction = construction
        circle.isFixed = fixed
        # return circle
        return sketch
    except:
        lib_util.disp_message("この３点では円は作成できません。")
        lib_util.disp_message("Failed:\n{}".format(traceback.format_exc()))
        return None


# /////////////////////////////////////////////////////////////
def create_two_point_rectangle(
    x1,
    y1,
    z1,
    x2,
    y2,
    z2,
    sketch=None,
    construction: bool = False,
    fixed: bool = False,
):
    """対角２点で長方形を描く"""
    if not sketch:
        sketch = lib_util.get_sketch()
    # kSketchPointで単位とスケッチ方向など調整
    p1 = kSketchPoint(x1, y1, z1, sketch)
    x1 = p1.sketch_x
    y1 = p1.sketch_y
    z1 = p1.sketch_z
    p2 = kSketchPoint(x2, y2, z2, sketch)
    x2 = p2.sketch_x
    y2 = p2.sketch_y
    z2 = p2.sketch_z

    try:
        # Create sketch rectangle
        sketchLines = sketch.sketchCurves.sketchLines
        sketchLines.isConstruction = construction
        sketchLines.isFixed = fixed
        startPoint = adsk.core.Point3D.create(x1, y1, z1)
        endPoint = adsk.core.Point3D.create(x2, y2, z2)
        lines = sketchLines.addTwoPointRectangle(startPoint, endPoint)
        line_0 = lines.item(0)
        line_1 = lines.item(1)
        line_2 = lines.item(2)
        line_3 = lines.item(3)
        line_0.isFixed = fixed
        line_1.isFixed = fixed
        line_2.isFixed = fixed
        line_3.isFixed = fixed
        line_0.isConstruction = construction
        line_1.isConstruction = construction
        line_2.isConstruction = construction
        line_3.isConstruction = construction
        # print(sketch.revisionId)
        return sketch
    except:
        lib_util.disp_message("この設定では長方形は作成できません。")
        lib_util.disp_message("Failed:\n{}".format(traceback.format_exc()))
        return None


# /////////////////////////////////////////////////////////////
def create_center_point_rectangle(
    x1,
    y1,
    z1,
    x2,
    y2,
    z2,
    sketch=None,
    construction: bool = False,
    fixed: bool = False,
):
    """中心指定で長方形を描く"""
    if not sketch:
        sketch = lib_util.get_sketch()
    # kSketchPointで単位とスケッチ方向など調整
    p1 = kSketchPoint(x1, y1, z1, sketch)
    x1 = p1.sketch_x
    y1 = p1.sketch_y
    z1 = p1.sketch_z
    p2 = kSketchPoint(x2, y2, z2, sketch)
    x2 = p2.sketch_x
    y2 = p2.sketch_y
    z2 = p2.sketch_z

    try:
        # Create sketch rectangle
        sketchLines = sketch.sketchCurves.sketchLines
        sketchLines.isConstruction = construction
        sketchLines.isFixed = fixed
        center_point = adsk.core.Point3D.create(x1, y1, z1)
        diagonal_point = adsk.core.Point3D.create(x2, y2, z2)
        lines = sketchLines.addCenterPointRectangle(center_point, diagonal_point)
        line_0 = lines.item(0)
        line_1 = lines.item(1)
        line_2 = lines.item(2)
        line_3 = lines.item(3)
        line_0.isFixed = fixed
        line_1.isFixed = fixed
        line_2.isFixed = fixed
        line_3.isFixed = fixed
        line_0.isConstruction = construction
        line_1.isConstruction = construction
        line_2.isConstruction = construction
        line_3.isConstruction = construction
        # print(sketch.revisionId)

        return sketch
    except:
        lib_util.disp_message("この設定では長方形は作成できません。")
        lib_util.disp_message("Failed:\n{}".format(traceback.format_exc()))
        return None
